use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Getopt::Long;
use feature "state";

my $gnuplot = 'C:\Program Files (x86)\gnuplot\bin\gnuplot';


######################################################################
# 
# FCAT - Frametime Capture Analysis Tool
#
######################################################################
# These are the colors used in the overlay
my @ref_colors = (
		 { r => 0xff, g=> 0xff, b=>0xff, cname => "White",},
		 { r => 0x00, g=> 0xff, b=>0x00, cname => "Lime",},
		 { r => 0x00, g=> 0x00, b=>0xff, cname => "Blue",},
		 { r => 0xff, g=> 0x00, b=>0x00, cname => "Red",},
		 { r => 0x00, g=> 0x99, b=>0x99, cname => "Teal",},
		 { r => 0x00, g=> 0x00, b=>0x80, cname => "Navy",},
		 { r => 0x00, g=> 0x99, b=>0x00, cname => "Green",},
		 { r => 0x00, g=> 0xff, b=>0xff, cname => "Aqua",},
		 { r => 0x80, g=> 0x00, b=>0x00, cname => "Marron",},
		 { r => 0xdc, g=> 0xdc, b=>0xdc, cname => "Silver",},
		 { r => 0x80, g=> 0x00, b=>0x80, cname => "Purple",},
		 { r => 0x80, g=> 0x80, b=>0x00, cname => "Olive",},
		 { r => 0xa0, g=> 0xa0, b=>0xa0, cname => "Gray",},
		 { r => 0xff, g=> 0x00, b=>0xff, cname => "Fuchsia",},
		 { r => 0xff, g=> 0xff, b=>0x00, cname => "Yellow",},
		 { r => 0xff, g=> 0xd0, b=>0x00, cname => "Orange",},
		 { r => 0x00, g=> 0x00, b=>0x00, cname => "---vblank",},              #vblank
		);


# Inputs
my $maxdis = 20;
my $blank_interval = 40;
my @infiles;
my @outfiles;
my @inf;
my $game = "";
my $fpsmode = 0;
my $plotfile1 = "plot1.plt";
my $plotfile2 = "plot2.plt";
my @gpus;
my @extras;

my $rr = 60;
my @start_frames;
my @end_frames;
my $ival = 1;
my $win = 10;
my $sper = .80;
my $debug = 0;
my $smode = "cum";
my $outfile;
my $runtsize = 20;
my $runtper = .25;
my $sample_colors = 0;


# plot commands
my $xmax;
my $ymax = 200;
my $gpu = "Unknown";
my $xmax2;
my $ymax2 = 200;
my $common;
my $config;
my @configfs;

my $result = GetOptions ("blankinterval=i" => \$blank_interval,          # Blank interval used in orignal extraction
		         "infile=s"   => \@infiles,                      # input file  
			 "outf=s"     => \@outfiles,			 

			 "config"     => \$config,                       # read a configuration file containing graphing options
			 "configfs"   => \@configfs,                     # Per data set coonfigurarion

			 "common=s"   => \$common,
		         "fraps=s"    => \@inf,                          # Fraps input file
		         "game=s"     => \$game,                         # game name
		         "gpu=s"      => \@gpus,                         # GPU name
		         "extra=s"    => \@extras,                       # extra run info (eg res)

		         "plot1=s"    => \$plotfile1,                    # File name used for gnuplot
		         "plot2=s"    => \$plotfile2,                    # File name used for gnuplot
		         "fps=i"      => \$fpsmode,                      # Run in fps mode vs frametime mode
		         "refresh=i"  => \$rr,                           # panel refresh rate
                         "startf=i"   => \@start_frames,                 # Video frame to start processing from
                         "endf=i"     => \@end_frames,                   # video frane to end processing at
			 "runt=i"     => \$runtsize,                     # Max number of scanlines in a runt
			 "runtper=f"  => \$runtper,                      # Max percentage of prior frame in a runt
                         "ival=f"     => \$ival,                         # Averaging period for summary stats (ms)
                         "xmax=s"     => \$xmax,                         # max xrange for gnuplot
                         "ymax=s"     => \$ymax,                         # max yrange for gnuplot
                         "xmax2=s"    => \$xmax2,                        # max xrange for gnuplot
                         "ymax2=s"    => \$ymax2,                        # max yrange for gnuplot
			 "samp"       => \$sample_colors,                # Force a sampling of colors at startup
                         "debug"      => \$debug);                       # Turn on messages


# open the global configuration file
sub get_configs {
    my ($filename, $ra_cdata) = @_;
    
    printf ("Processing configuration file....\n");
    open FILE, $filename or die "couldn't open '$filename' for read";

    my @file = <FILE>;
    close FILE;
 

    # format of config file is :
#    gpu_regex,  gaame_regex,  other_regex : option_name -> override_value

   
    # Load in file (tab or comma seperated)
    @$ra_cdata = map { chomp; my @d = split /\s*[\,\t]\s*/; \@d; } @file;
}



# Loop through input files
my $ofi = 0;
my $start_frame = 0;
my $end_frame;

open PLOT, "> $plotfile1";
my $of;
if ($common) { 
    $of = "$common";
}
#
# Generate the plots using gnuplot
#
my $title;

if ($game) {
    $title = $game;
} else {
    $title = $infiles[0];
}

plot1_header ($of,  $xmax, $ymax, $title);    

my $plots =0;
my $extra = "";

foreach my $infile (@infiles) {
    # The array of colors generated by sampling 1st 16 bars
    if (exists $outfiles[$ofi]) {
	$outfile = $outfiles[$ofi];
    }

    if (exists $gpus[$ofi]) {
	$gpu = $gpus[$ofi];
    }

    if (exists $extras[$ofi]) {
	$extra = $extras[$ofi];
    }

    if (exists $start_frames[$ofi]) {
	$start_frame = $start_frames[$ofi];
    }

    if (exists $end_frames[$ofi]) {
	$end_frame = $end_frames[$ofi];
    }

    $ofi++;

    # Generate an output filename
    unless ($outfile) { 
	$outfile = $infile;
    }
    
    my $logfile = $outfile . "\.log";
    my $issuefile = $outfile . "\.issues";

    # Open the logs
    open LOGF, "> $logfile";
    open ISF, "> $issuefile";

    # Read the Inputfile
    my @data;
    readdata($infile, \@data);
    
    # Parse the datafile
    my @newdata;
    my $rc = parse_datafile(\@data, \@newdata);
    if ($rc) {
	next;
    }

    dump_raw(\@newdata, "$outfile.hw");

    # Check data and calculate some statistics
    my %stats;
    check_data(\@newdata, $outfile, \%stats);
    dump_stats (\%stats, $outfile, $infile);
    dump_raw_stats (\%stats, "$outfile.hw.stats");

    # Add this GPUs data
    plot1 ($plots, "$outfile.hw", $gpu, $extra);

    #increment
    $plots++;

    # Run plot2 for each input
    plot2 ("$outfile.hw.stats", "$outfile.stats", $plotfile2, $xmax2, $ymax2, $title, $gpu, $extra);
    
    my $rc2 = system ($gnuplot, $plotfile2);
    if ($rc2) {
	printf ("Error running GNUPLOT\n");
    }
    
    # cleanup
    printf LOGF ("finished\n");
    close LOGF;
    close ISF;

}

# Run the common plot
printf PLOT ("\n\nquit\n");
close PLOT;

my $rc = system ($gnuplot, $plotfile1);
if ($rc) {
    printf ("Error running GNUPLOT\n");
}

exit(0);


#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
#
# Read the entire datafile
#
#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
sub readdata {
    my ($filename, $ra_data) = @_;
    
    printf ("Processing $filename....\n");
    printf LOGF ("Processing $filename....\n");
    open FILE, $filename or die "couldn't open '$filename' for read";

    my @file = <FILE>;
    close FILE;
    
    shift @file; # skip header
    
    # Load in file (tab or comma seperated)
    @$ra_data = map { chomp; my @d = split /\s*[\t]\s*/; \@d; } @file;
}


#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
#
# Process bars
#
#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
sub parse_datafile {
    my ($ra_data, $ra_newdata) = @_;

    my @ov_colors = ();
    my $sindex = 0;
    
    # Some local parsing variables
    my $last_refresh = 0;
    my $real_time = 0;
    my $scan_lines = 0;
    my $mode_change = 0;
    my $drop_cnt = 0;

    my $cur_stripe_color = 0;
    my $cur_stripe_length = 0;
    my $cur_scan_lines = 0;

    # Remember last colors
    my $last_name = "Init";
    my $last_index = 17;
    my $last_color = "Init";

    # Look up the color expected (assuming just incrementing overlay colors)
    my $want_index = 17;
    my $want_name = "Init";

    # some status
    my $sync_lost = 0;
    my $a_ref;
    my $vblank = 0;
    my $hold_over = 0;
    
    my $fatal = 0;

    # Step through each line  format is :
    # [frame]
    # [scanlines]
    # [time (ms)]
    # [fps]	
    # [frame start (s)]	
    # [screen refresh]
    # [color]
    while ($a_ref = shift @$ra_data) {
	my $framenum = @$a_ref[0];
	my $lines= @$a_ref[1];
	my $time = @$a_ref[2];
	my $fps = @$a_ref[3];
	my $frame_start = @$a_ref[4];
	my $refresh_num = @$a_ref[5];
	my $color = hex @$a_ref[6];

	# Check for the comma in some fields
	$time =~ s/\,/\./;
	$fps =~ s/\,/\./;
	$frame_start =~ s/\,/\./;
	
	# Skip this display frame if out of bounds
	next if ($refresh_num < $start_frame);
	next if ($end_frame and ($refresh_num > $end_frame));
	
	#----------------------------------------------------------------------------------------
	#
	# Process scanline counting : how many scanlines per display frame?
	#
	#----------------------------------------------------------------------------------------

	# Detecting a new "display frame".  - Check last frame's scan line count
	if ($refresh_num != $last_refresh) {
	    printf LOGF ("\n++++ FRAME (%10d) +++++++++++++++++++++++++++++++++++++++++++++++++++\n\n", $last_refresh); 

	    # Record number of scan lines in this sample	    
	    if ($scan_lines ) {
		if ($scan_lines != $cur_scan_lines) {
		    printf ISF ("Error : Unmatched scanlines on frame $last_refresh : $scan_lines <> $cur_scan_lines\n");
		    printf LOGF ("Error : Unmatched scanlines on frame $last_refresh : $scan_lines <> $cur_scan_lines\n");
		    printf ("Error : Unmatched scanlines on frame $last_refresh : $scan_lines <> $cur_scan_lines\n");
		}
	    } else {
		# Save number of scanlines in this capture - this only gets set once.
		$scan_lines = $cur_scan_lines;
		printf LOGF ("$scan_lines scanlines (including blanking interval) detected in capture.\n");
	    }
	    
	    # Update to new rerfresh frame
	    $last_refresh = $refresh_num;
	    $cur_scan_lines = $lines;
	    
	} else {
	    
	    # Add to the current scan line count
	    $cur_scan_lines += $lines;
	}


	#----------------------------------------------------------------------------------------
	#
	# Now look at bar lengths to determine rendered frame times
	#
	#----------------------------------------------------------------------------------------
	#
	# A few cases
	# 0 - Unknown color - assumit it is dithered and add to the next bar
	# 1 - Vblank - a vertical blank is detected - just rememeber it.
	# 2 - NV Scanout Bug
	# 3 - bar is a same a last color (prior bar should have been a vblank)
	# 4 - bar is a different color - and incorrect (dropped frame)
	# 5 - bar is a different color - and correct
	#

	# Look up the current color and color index
	my ($cur_index, $cur_name) = find_color($maxdis, $color, \@ov_colors, \$sindex, $lines);

	#----------------------------------------------------------------------------------------
	# Case 0 : If the color is Unknown, then just add the length to the next stripe
	#----------------------------------------------------------------------------------------
	if ($cur_index == 17) {
	    # If you get 0x7f0000 color test is over
	    last if ($color == 0x7f0000);

	    printf ("WARNING : Unknown color @ $refresh_num (%5.2f): color : 0x%06x $lines - adding to next frame.\n", $frame_start, $color);
	    printf LOGF ("WARNING : Unknown color @ $refresh_num (%5.2f): color : 0x%06x $lines - adding to next frame.\n", $frame_start, $color);
	    $hold_over += $lines;
	    
	#----------------------------------------------------------------------------------------
	# Case 1 : Vblank
	#----------------------------------------------------------------------------------------
	# Check for a vblank - just remember vblank size and continue.
	} elsif ($cur_name eq "---vblank") {
	    if ($lines == $blank_interval) {
		$vblank = $lines;
	    } else {
		unless ($mode_change) {
		    printf ISF ("Likely Mode change @ $refresh_num : Mismatched vblank size expect $blank_interval, got $lines\n");
		    printf LOGF ("Likely Mode change @ $refresh_num : Mismatched vblank size expect $blank_interval, got $lines\n");
		    printf ("Likely Mode change @ $refresh_num : Mismatched vblank size expect $blank_interval, got $lines\n");
		    $mode_change = 1;
		}
	    }
	#----------------------------------------------------------------------------------------
	# Case 3 : Continued color.  Must have a vblank in between
	#----------------------------------------------------------------------------------------
 	# Check for a continued colored bar (note that last_index is not updated for vblanks)
	} elsif ($cur_index == $last_index) {
	    
	    # this must be a continued bar accross a vblank - Add the vblank time to the rendered frame length
	    $cur_stripe_length += $lines + $vblank;
	    
	    # If there was no vblank then something is hosed.
	    if ($vblank == 0) {
		printf LOGF ("          --Continued color detected : $cur_name\n");
		$vblank = 0
	    } else {
		# This is a near cousin color
		printf LOGF ("Near Cousin : %5d (%5.2f): color : %6s length = $lines\n", $refresh_num, $frame_start, $cur_name);
	}

	#----------------------------------------------------------------------------------------
	# Case 4 : New color...and wrong (dropped frame)
	#----------------------------------------------------------------------------------------
        # A Check for wrong color sequence
	}  else {
	    if (($cur_index != $want_index) and ($want_index != 17)) {
		
		# Wrong color in initial sequence
		unless ($want_name) {
		    $want_name = "init";
		}
		
		if ($drop_cnt <10 and not $fatal) {
		    printf ("Error (%6d, %5.2fs) : DROPPED FRAME - Expected index $want_index : %6s, got index $cur_index %6s: length = $lines\n", $refresh_num, $frame_start, $want_name, $cur_name);
		}

		if ($drop_cnt == 10) {
		    printf ("More dropped frames not displayed for clarity.\n");
		}

		printf ISF ("Error (%6d, %5.2fs) : DROPPED FRAME - Expected index $want_index : %6s, got index $cur_index %6s: length = $lines\n", $refresh_num, $frame_start, $want_name, $cur_name);
		printf LOGF ("Error (%6d, %5.2fs) : DROPPED FRAME - Expected index $want_index : %6s, got index $cur_index %6s: length = $lines\n", $refresh_num, $frame_start, $want_name, $cur_name);

		$drop_cnt++;
		
		# Add a 0 length record for all missing frames
		my $i = $want_index;
		my $n = $want_name;

		unless ($refresh_num < 16) {
		    do {
			push @$ra_newdata, {dframe => $refresh_num, frame => $framenum, scanlines => 0, time => $frame_start, color => $n, frame_time => 0, fps => 100000 };
#		    printf ("                      - Inserted 0s record : index $i for %6s\n", $n);
			printf ISF ("                      - Inserted 0s record : index $i for %6s\n", $n);
			($i, $n) = next_color($i, \@ov_colors);
			unless ($n) { $n= "undef";}

		    } until ($i == $cur_index)
		} else {
		    if (not $fatal) {
			printf ("Bad Trace - Fatal Error\n");
			$fatal = 1;
		    }
	        }

	    } else {
		printf LOGF ("          --Sequence validated $cur_index $cur_name\n"); 
	    }

	    #----------------------------------------------------------------------------------------
	    # Case 5 : New color...and correct (after shifting above for dropped frames
	    #----------------------------------------------------------------------------------------
	    
	    # Add in any holdover
	    $cur_stripe_length += $hold_over;
	    $hold_over = 0;
	    
	    # If the prior line was a vblank then a color change happened during the vblank.  split the difference
	    if ($vblank) {
		$cur_stripe_length += $vblank/2;
		$lines += $vblank/2;

		$vblank = 0;
	    }
	    
	    #
	    # Process the prior bar - and save the data
	    #
	    # Calculate frame time
	    my $ft = 0;
	    my $fp= 0;
	    
	    # if we have calculated number of scanlines in sample
	    if ($scan_lines) {
		
		# Calculate the frame time by looking at the length of the bar, the total # of lines in a frame and the refresh rate.
		$ft = $cur_stripe_length / $scan_lines / $rr;
		
		# convert to fps
		$fp = 1/$ft;
	    }
	    
	    # Record length of last line
	    unless ($last_index == 17) { 
		push @$ra_newdata, {dframe => $refresh_num, frame => $framenum, scanlines => $cur_stripe_length, time => $frame_start, color => $last_name, frame_time => $ft*1000, fps => $fp };
		printf LOGF ("          --Saving on Frame $refresh_num  idx %2d color %8s (0x%06x), scanlines %5d\n", $last_index, $last_name, $last_color, $cur_stripe_length); 
	    }
	    
	    # Keep track of last valid bar colors
	    $last_name = $cur_name;
	    $last_index = $cur_index;
	    $last_color = $color;
	    $cur_stripe_length = $lines;

	    # Update the new target colors - increment
	    ($want_index, $want_name) = next_color($last_index, \@ov_colors);

	}
    }
    
    return($fatal);

}

#
# Given an RGB value find the closest color in the overlay list
#


sub find_color {
    my ($max, $c, $ra_ov, $ri_sindex, $lines) = @_;

    # Extract RGB
    my $r = ($c >> 16) & 0xff;; 
    my $g = ($c >> 8) & 0xff;; 
    my $b = $c & 0xff;; 
    
    # Scan the array to find closest match or "unknown" if out of range
    my $i;
    my $closest = 10000;
    my $found = 17;
    my $name = "?";
    
    printf LOGF ("### Looking for -> %02x %02x %02x with $lines scanlines\n", $r, $g, $b, $lines);
    
    # specail case vblank
    if ($c == 0) {
	return (16, "---vblank");	
    }

    # see if the color is in the index.
    $closest = 10000;
    $found = 17;
    my $dis = 0;
    for ($i = 0; $i < $$ri_sindex; $i++) {
	my $rh_co = $ra_ov->[$i];
	
	$dis = closeto ($rh_co, $r, $g, $b);	
#	printf LOGF ("Checking Exisiting -> $i, %8s, %6.2f\n", $rh_co->{cname}, $dis);
	
	if ($dis < $closest) {
	    $closest = $dis;
	    $found = $i;
	    $name = $rh_co->{cname};
	}
    }

    # If the distance is within tolerance return the color name and index
    if ($closest < $max) {
	printf LOGF ("### Found -> $found, %8s, %6.2f\n", $name, $closest);
	return ($found, $name);
    }

    printf LOGF ("### Color not in index : Dis = $dis, MAX = $max\n");

    # first update the ov_array if this is a new color
    if ($$ri_sindex <16) {
	
	# If very short number of lines ignore during intial color detection
	if ($lines < 2) {
	    return (17, "Unknown");
	}
	
	# See if we can find an english name for this value
	for (my $i = 0; $i < 16; $i++) {
	    my $rh_co = $ref_colors[$i];
	    
	    my $dis = closeto ($rh_co, $r, $g, $b);	
	    printf LOGF ("Looking for Color Name match -> $i, %8s, %6.2f\n", $rh_co->{cname}, $dis);
	    
	    if ($dis < $closest) {
		$closest = $dis;
		$found = $i;
		$name = $rh_co->{cname};
	    }
	}
	

	# Use the closest name < $target
	my $cn = "samp".$$ri_sindex;
	if ($closest < 100) {
	    $cn = $name;
	} 
	
	# record the current color
	$ra_ov->[$$ri_sindex] =  { 
	    r => $r, 
	    g => $g, 
	    b => $b,
	    cname => $cn
	};

	printf LOGF ("### -----> New Color  -> %3d, %06x, %8s\n", $$ri_sindex, $c,$cn);

	# move onto the next sample color
	$$ri_sindex++;
	
	return ($$ri_sindex -1, $cn);

    } 

    return (17, "Unknown");
}

sub closeto {
    my ($rh, $r, $g, $b) = @_;
    
    unless ($rh) {
	return 10000;
    }

    my $cr = $rh->{r};
    my $cg = $rh->{g};
    my $cb = $rh->{b};

    my $dis = ($r - $cr) **2;
    $dis += ($g - $cg) **2;
    $dis += ($b - $cb) **2;
    $dis = sqrt ($dis);
    return ($dis);
}

# What is the correct next color index and color
sub next_color {
    my ($cur, $ra_ov) = @_;
    
    my $i = $cur+1;

    # Wrap if > 15
    if ($i > 15) {
	$i = 0;
    }
    
    # Retrun index and color
    my $rh_co = $ra_ov->[$i];

    return ($i, $rh_co->{cname});
}



sub max { return ($_[0] > $_[1]) ? $_[0] : $_[1]; }
sub min { return ($_[0] < $_[1]) ? $_[0] : $_[1]; }

#---------------------------------------------------------
#
# Write the RAW processed bar lengths
#
#---------------------------------------------------------
sub dump_raw {
    my ($ra_newdata, $outf) = @_;
    
    #
    # Write out the extracted bar lengths
    #
    open DATA, ">$outf" or die "couldn't open '$outf'";
    if ($fpsmode) {
	print DATA join "\n", map { "$_->{time}" . "\t" .  "$_->{fps}" } @$ra_newdata;
    }
    else {
	print DATA join "\n", map { "$_->{time}" . "\t" .  "$_->{frame_time}" } @$ra_newdata;
    }
    
    close DATA;
}  
 

#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
#
# Check data for artifacts and generate some averages
#
#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
sub check_data {
    
    my ($ra_newdata, $outf, $ra_stats) = @_;
  
    #
    # Count how many frames are completed during an interval
    #
    my @window;
    my @window_sl;
    my $cnt = 0;
    my $carry = 0;
    my $frame = 0;
    my $slice = 0;
    my $first = -1;
    my $last = 0;
    my $last_scanlines = 1;

    open HWFTS, ">$outf.frametimes.csv" or die "couldn't open '$outf.frametimes.csv'";
    printf HWFTS ("Frame,Time (ms)\n");

    # Iterate across all extracted frame lengths
    foreach (@$ra_newdata){
	my $refresh_num = $_->{dframe};
	my $framenum = $_->{gframe};
	my $scanlines = $_->{scanlines};
	my $ft = $_->{frame_time};
	my $t = $_->{time};
	my $fl = $_->{fps};
	my $c = $_->{color};

	# Detection variables
	my $runt = 0;
	my $drop = 0;

	# Skip this display frame if out of bounds
	next if ($refresh_num < $start_frame);
	next if ($end_frame and ($refresh_num > $end_frame));

	# Calculate a runt as a percentage or below a min nunmber of scanlines
	my $runtcheck =  $scanlines/$last_scanlines;
	my $runtdet = (($runtcheck > 0) and ($runtcheck < $runtper));
	if ($scanlines < $runtsize) {
	    $runtdet = 1;
	}
	    
	# Check for a drop
	if ($ft == 0 and ($scanlines == 0)) {
	    $drop = 1;
#	    printf      ("INSERTED FRAME FOR drop     at time : %5.2f s. Frame (%5d), scanlines: %5d %7s, Frametime: %5.2f ms.\n", $t, $refresh_num, $scanlines, $c, $ft);	    
	    printf LOGF ("INSERTED FRAME FOR drop     at time : %5.2f s. Frame (%5d), scanlines: %5d %7s, Frametime: %5.2f ms.\n", $t, $refresh_num, $scanlines, $c, $ft);	    
	    printf ISF  ("INSERTED FRAME FOR drop     at time : %5.2f s. Frame (%5d), scanlines: %5d %7s, Frametime: %5.2f ms.\n", $t, $refresh_num, $scanlines, $c, $ft);	    

	# Check for a runt1
	} elsif ($runtdet)  {
            printf ISF  ("RUNT FRAME %7.2f%%        at time : %5.2f s. Frame (%5d), scanlines: %5d %7s, Frametime: %5.2f ms.\n", $runtcheck*100, $t, $refresh_num, $scanlines, $c, $ft);
            printf LOGF ("RUNT FRAME %7.2f%%        at time : %5.2f s. Frame (%5d), scanlines: %5d %7s, Frametime: %5.2f ms.\n", $runtcheck*100, $t, $refresh_num, $scanlines, $c, $ft);
#            printf      ("RUNT FRAME %7.2f%%        at time : %5.2f s. Frame (%5d), scanlines: %5d %7s, Frametime: %5.2f ms.\n", $runtcheck*100, $t, $refresh_num, $scanlines, $c, $ft);
	    $carry += $ft;
	    $runt = 1;

	} else {
	    # Add a record to the frametime file unless to short
	    $frame++;
	    printf HWFTS ("%5d, %10.3f\n", $frame, $ft + $carry);
	    $carry = 0;
	}

	

	$cnt++;
	if ($scanlines) {
	    $last_scanlines = $scanlines;
	}
	
	unshift (@window, $ft);
	unshift (@window_sl, $scanlines);

	#-----------------------------------------------------------
	# Calculate some bucketed counts
	#-----------------------------------------------------------
	# First find the correct bucket
	while ($t >= ($slice * $ival)) {
	    $slice++;
	}

	# Initial the counters at this slice.
	my $ptr = $slice - 1;

	unless (exists $ra_stats->{OFPS}[$ptr])   { $ra_stats->{OFPS}[$ptr] = 0};
	unless (exists $ra_stats->{FPS}[$slice -1])    { $ra_stats->{FPS}[$slice -1] = 0};
	unless (exists $ra_stats->{DROPS}[$slice -1])  { $ra_stats->{DROPS}[$slice -1] = 0};
	unless (exists $ra_stats->{RPS}[$slice -1])    { $ra_stats->{RPS}[$slice -1] = 0};

	# Update intervals counts (FPS)
	$ra_stats->{OFPS}[$slice-1]++;       

	# Only count "rea" FPS that is not a runt or a drop
	if ($runt) { 
	    $ra_stats->{RPS}[$slice-1]++; 
	} elsif ($drop) { 
	    $ra_stats->{DROPS}[$slice-1]++; 
	} else {
	    $ra_stats->{FPS}[$slice-1]++;       
	}
	

	# Remember the first slice
	unless ($first >= 0) {
	    $first = $slice -1;
	}

	# Increment the number of frames completed this interval
	$last = $slice-1;
    }

    $ra_stats->{FIRST} = $first;
    $ra_stats->{LAST} = $last;

    close HWFTS;

}


#---------------------------------------------------------
#
# Write the stats files
#
#---------------------------------------------------------
sub dump_stats {
    my ($ra_stats, $outf, $infile) = @_;
    
    # Open up the files
    open HWFPS, ">$outf.fps.csv" or die "couldn't open '$outf.fps.csv'";
    printf HWFPS ("FPS\n");

    # Print a title line in the csv file
    open HWSPS, ">$outf.sps.csv" or die "couldn't open '$outf.sps.csv'";
    printf HWSPS ("Time,HWFPS,Drop,Runt,RAW_FPS\n");

    # Dump the completed counts
    my $j;

    my $fps_sum = 0;
    my $ofps_sum =0;
    my $sps_sum =0;
    my $runt_sum =0;
    my $drop_sum =0;

    # Get range of slices
    my $last = $ra_stats->{LAST};
    my $first = $ra_stats->{FIRST};
    my $slices = $last - $first;
    
    for ($j = $first; $j<$last; $j++) {
	# Print the time
	printf HWSPS ("%7d, ", $j);

	if ($ra_stats->{FPS}[$j]) {
	    printf HWFPS ("%.2f\n", $ra_stats->{FPS}[$j]/$ival);
	    printf HWSPS ("%.2f, ", $ra_stats->{FPS}[$j]/$ival);
	    $fps_sum += $ra_stats->{FPS}[$j]/$ival;
	} else {
	    printf HWSPS ("%.2f, ", 0);
	}
	
	# Show drops
	if ($ra_stats->{DROPS}[$j]) {
	    printf HWSPS ("%.2f, ", $ra_stats->{DROPS}[$j]/$ival);
	    $drop_sum += $ra_stats->{DROPS}[$j]/$ival;
	} else {
	    printf HWSPS ("%.2f, ", 0);
	}

	# Show runts
	if ($ra_stats->{RPS}[$j]) {
	    printf HWSPS ("%.2f,", $ra_stats->{RPS}[$j]/$ival);
	    $runt_sum += $ra_stats->{RPS}[$j]/$ival;
	} else {
	    printf HWSPS ("%.2f,", 0);
	}

	# Show original FPS
	if ($ra_stats->{OFPS}[$j]) {
	    printf HWSPS ("%.2f,", $ra_stats->{OFPS}[$j]/$ival);
	    $ofps_sum += $ra_stats->{OFPS}[$j]/$ival;
	} else {
	    printf HWSPS ("%.2f,, ", 0);
	}

    }

    # Generate a summary file
    open LOGS, "> $outf.sum";
    my @timedate = localtime(time);
    my $year = $timedate[5] + 1900;
    my $mon = $timedate[4]+1;
    my $day = $timedate[3];

    my $h = $timedate[2];
    my $m = $timedate[1];
    my $s = $timedate[03];

    # Calculating some averages
    printf LOGS ("%4d-%02d-%02d %2d:%02d:%02d - $infile\n", $year, $mon, $day, $h, $m, $s);
    printf LOGS ("Calculated FPS - %5.2f - Original FPS %5.2f\n", $fps_sum/$slices, $ofps_sum/$slices);
    printf LOGS ("Calculated Frames (True/Runts/Drops): %5.0f / %5.0f / %5.0f\n", $fps_sum, $runt_sum, $drop_sum, $sps_sum);
    printf LOGS ("Original Frames (FRAPS): %5.0f   Time : $slices sec.\n\n", $ofps_sum);

    # 
    close LOGS;
    close HWSPS;
    close HWFPS;
}



#---------------------------------------------------------
#
# Write the RAW processed bar lengths
#
#---------------------------------------------------------
sub dump_raw_stats {
    my ($ra_stats, $outf) = @_;
    
    #
    # Write out the extracted bar lengths
    #
    open DATA, ">$outf" or die "couldn't open '$outf'";
    my $first = $ra_stats->{FIRST};
    my $last = $ra_stats->{LAST};

    my $s;
    for ($s = $first; $s < $last; $s++) {

	# Some shorthand
        my $fps = $ra_stats->{FPS}[$s];
        my $r = $ra_stats->{RPS}[$s];
        my $d = $ra_stats->{DROPS}[$s];
        my $ofps = $ra_stats->{OFPS}[$s];

	printf  DATA ("%.2f\t%.2f\t%.2f\t%.2f\t%.2f\n", $s+1, $fps, $fps+$r, $fps+$r+$d, $ofps);
    }

    close DATA;
}  
 




# Print the fancy gnuplot graph
sub plot1_header {
	    
    my ($outf, $xm, $ym, $title) = @_;

    # Auto axis?
    my $xauto = "set autoscale x";
    my $xrange = "";
    if ($xm) {
	$xauto = "";
	$xrange = "set xrange [0:$xm]";
    }
    
    my $yauto = "set autoscale x";
    my $yrange = "";
    if ($ym) {
	$yauto = "";
	$yrange = "set yrange [0:$ym]";
    }
    
    # Insert plot setup into file
    print PLOT << "EOT";
set term pngcairo size 1920, 1080
set xlabel "Scene time (s)"
set style fill transparent solid 0.5 noborder
$xauto
$xrange
$yauto
$yrange
set output '$outf.png'
set title '$title' font "Sans,40"
set style line 1 lt 1 lw 1 pt 3 linecolor rgb "red" 
set style line 2 lt 1 lw 1 pt 3 linecolor rgb "brown"
set style line 3 lt 1 lw 1 pt 3 linecolor rgb "yellow"
set style line 4 lt 1 lw 1 pt 3 linecolor rgb "pink"
set style line 5 lt 1 lw 1 pt 3 linecolor rgb "#336600"
set style line 6 lt 1 lw 1 pt 3 linecolor rgb "black"
set style line 7 lt 1 lw 1 pt 3 linecolor rgb "blue"
set style line 8 lt 1 lw 1 pt 3 linecolor rgb "green"

set grid ytics lt 0 lw 1 lc rgb "#880000"
set grid xtics lt 0 lw 1 lc rgb "#880000"


EOT
    
    
    if ($fpsmode) {
	printf PLOT ("set ylabel \"FPS\"\n");
    } else {
	printf PLOT ("set ylabel \"Frametime (ms)\"\n");
    }

}



# Print the fancy gnuplot graph
sub plot1 {
	    
    my ($cnt, $datafile, $gpu, $extra) = @_;

    # Pick a line type
    state $amd = 0;
    state $nv = 0;

    my $hwlt = 1;

    unless ($gpu) { $gpu = "Default"};

    if ($cnt == 0) {
	printf PLOT ("plot ");
    } else {
	printf PLOT ("\, \\\n     ");
    }
    
    # Pick NV or ATI colors
    if ($gpu =~ /^gtx/i) {
	$hwlt = 5 + $nv++;
    } elsif ($gpu =~ /^gk/i) {
	$hwlt = 5 + $nv++;
    } else {
	$hwlt = 1 + $amd++;
    }
    
    # Add HW plot
    printf PLOT ("\'$datafile\' using 1:2 ls $hwlt ti \"$gpu $extra\" with lines");


}



# Print the fancy gnuplot graph
sub plot2 {

    my ($datafile, $outf, $plotfile, $xm, $ym, $title, $gpu, $extra) = @_;

    ## Build up a plotfile for GNUPLOT
    open PLOT2, "> $plotfile";

    # Auto axis?
    my $xauto = "set autoscale x";
    my $xrange = "";
    if ($xm) {
	$xauto = "";
	$xrange = "set xrange [0:$xm]";
    }
    
    my $yauto = "set autoscale y";
    my $yrange = "";
    if ($ym) {
	$yauto = "";
	$yrange = "set yrange [0:$ym]";
    }

    # Insert plot setup into file
    print PLOT2 << "EOT";
set term pngcairo size 1920, 1080
set xlabel "Scene time (s)"
$xauto
$xrange

set ylabel "FPS"
$yauto
$yrange


set boxwidth 0.3 relative

set output '$outf.png'
set title '$title - $gpu $extra'  font "Sans,40"

set style line 1 lt 1 lw 1 pt 3 linecolor rgb "red" 
set style line 2 lt 1 lw 1 pt 3 linecolor rgb "orange"
set style line 3 lt 1 lw 1 pt 3 linecolor rgb "green"
set style line 4 lt 1 lw 5 pt 3 linecolor rgb "blue"
set style line 5 lt 1 lw 5 pt 3 linecolor rgb "black"

set grid ytics lt 0 lw 1 lc rgb "#880000"
set grid xtics lt 0 lw 1 lc rgb "#880000"

EOT

    
    unless ($gpu) { $gpu = "Default"};
    
    # Add HW plot
    printf PLOT2 ("plot \'$datafile\' using 1:2:3 ls 2 ti \"RUNTS\" with filledcurves, ");
    printf PLOT2 ("\'$datafile\' using 1:3:4 ls 1 ti \"DROPS\" with filledcurves, ");
    printf PLOT2 ("\'$datafile\' using 1:4:5 ls 3 notitle with filledcurves, ");
    printf PLOT2 ("\'$datafile\' using 1:5 ls 5 ti \"FRAPS\" with lines, ");
    printf PLOT2 ("\'$datafile\' using 1:2 ls 4 ti \"FPS\" with lines ");


    printf PLOT2 ("\nquit\n");
    close PLOT2;


}







